from django import forms
from .models import Enquiry, CareerApplication

class EnquiryForm(forms.ModelForm):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        for name, field in self.fields.items():
            css = field.widget.attrs.get("class", "")
            field.widget.attrs["class"] = (css + " input").strip()

    class Meta:
        model = Enquiry
        fields = [
            "full_name", "phone", "email",
            "enquiry_type", "location", "preferred_contact_time",
            "message"
        ]
        widgets = {
            "message": forms.Textarea(attrs={"rows": 5}),
        }

class CareerApplicationForm(forms.ModelForm):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        for name, field in self.fields.items():
            css = field.widget.attrs.get("class", "")
            field.widget.attrs["class"] = (css + " input").strip()

    class Meta:
        model = CareerApplication
        fields = ["position", "full_name", "phone", "email", "cv", "notes"]
        widgets = {
            "notes": forms.Textarea(attrs={"rows": 4}),
        }


from .models import CareRequest, CaregiverProfile, CareAssignment

class CareRequestStep1Form(forms.Form):
    full_name = forms.CharField(max_length=140)
    phone = forms.CharField(max_length=30)
    email = forms.EmailField(required=False)
    location = forms.CharField(max_length=160, required=False)

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        for _, field in self.fields.items():
            field.widget.attrs["class"] = (field.widget.attrs.get("class","") + " input").strip()

class CareRequestStep2Form(forms.Form):
    request_type = forms.ChoiceField(choices=CareRequest.REQUEST_TYPES)
    patient_age = forms.IntegerField(required=False, min_value=0)
    care_needs = forms.CharField(widget=forms.Textarea(attrs={"rows": 4}))
    schedule_notes = forms.CharField(max_length=220, required=False)

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        for _, field in self.fields.items():
            field.widget.attrs["class"] = (field.widget.attrs.get("class","") + " input").strip()

class CareRequestStep3Form(forms.Form):
    preferred_start_date = forms.DateField(required=False, widget=forms.DateInput(attrs={"type":"date"}))
    preference_notes = forms.CharField(required=False, widget=forms.Textarea(attrs={"rows": 3}))
    risk_notes = forms.CharField(required=False, widget=forms.Textarea(attrs={"rows": 3}))

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        for _, field in self.fields.items():
            field.widget.attrs["class"] = (field.widget.attrs.get("class","") + " input").strip()

class CaregiverAvailabilityForm(forms.ModelForm):
    class Meta:
        model = CaregiverProfile
        fields = ["phone", "location", "skills", "availability", "is_active"]
        widgets = {
            "skills": forms.Textarea(attrs={"rows": 3}),
            "availability": forms.Textarea(attrs={"rows": 3}),
        }

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        for _, field in self.fields.items():
            css = field.widget.attrs.get("class", "")
            field.widget.attrs["class"] = (css + " input").strip()

class AssignmentCreateForm(forms.ModelForm):
    class Meta:
        model = CareAssignment
        fields = ["care_request", "caregiver", "status", "start_date", "end_date", "notes"]
        widgets = {"notes": forms.Textarea(attrs={"rows": 3}),
                   "start_date": forms.DateInput(attrs={"type":"date"}),
                   "end_date": forms.DateInput(attrs={"type":"date"})}

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        for _, field in self.fields.items():
            css = field.widget.attrs.get("class", "")
            field.widget.attrs["class"] = (css + " input").strip()


# -----------------------------------------------------------------------------
# Support Ticket Forms
# -----------------------------------------------------------------------------
class SupportTicketForm(forms.Form):
    full_name = forms.CharField(max_length=120, widget=forms.TextInput(attrs={"placeholder":"Your name"}))
    phone = forms.CharField(max_length=40, required=False, widget=forms.TextInput(attrs={"placeholder":"Phone (optional)"}))
    email = forms.EmailField(required=False, widget=forms.EmailInput(attrs={"placeholder":"Email (optional)"}))
    subject = forms.CharField(max_length=180, required=False, widget=forms.TextInput(attrs={"placeholder":"Subject (optional)"}))
    message = forms.CharField(widget=forms.Textarea(attrs={"placeholder":"How can we help?"}))
